/**
 * jDownloads Admin Monitoring Modal JavaScript
 * @package mod_jdownloads_admin_monitoring
 * @version 4.1
 */

(function(document, window) {
    'use strict';
    
    class JDownloadsMonitoring {
        constructor(config) {
            this.config = config;
            this.modalElement = document.getElementById(config.modalId);
            this.modal = null;
            this.iframe = null;
            this.loader = null;
            this.modalTitle = null;
            
            this.init();
        }
        
        init() {
            if (!this.modalElement) {
                console.error('JDownloads Monitoring: Modal element not found');
                return;
            }
            
            // Initialize Bootstrap Modal
            this.modal = new bootstrap.Modal(this.modalElement);
            this.iframe = document.getElementById(this.config.modalId + '-iframe');
            this.loader = document.getElementById(this.config.modalId + '-loader');
            this.modalTitle = document.getElementById(this.config.modalId + 'Label');
            
            // Attach event handlers
            this.attachHandlers();
        }
        
        attachHandlers() {
            // Handle modal link clicks
            document.querySelectorAll('.jd-monitoring-modal-link').forEach(link => {
                link.addEventListener('click', (e) => this.handleLinkClick(e));
            });
            
            // Reset modal on close
            this.modalElement.addEventListener('hidden.bs.modal', () => this.resetModal());
            
            // Handle iframe load events
            if (this.iframe) {
                this.iframe.addEventListener('load', () => this.onIframeLoad());
                this.iframe.addEventListener('error', () => this.onIframeError());
            }
        }
        
        handleLinkClick(e) {
            e.preventDefault();
            
            const link = e.currentTarget;
            const mode = link.dataset.mode;
            const test = link.dataset.test || '0';
            const modeName = link.textContent.trim();
            
            // Build scan URL
            const url = this.buildScanUrl(mode, test);
            
            // Update modal title with mode name
            if (this.modalTitle) {
                this.modalTitle.innerHTML = '<i class="icon-eye"></i> ' + this.escapeHtml(modeName);
            }
            
            // Load monitoring in modal
            this.loadMonitoring(url);
        }
        
        buildScanUrl(mode, test) {
            const params = new URLSearchParams({
                key: this.config.secretKey,
                mode: mode,
                test: test,
                modal: '1'  // Flag for modal display
            });
            
            return this.config.baseUrl + '?' + params.toString();
        }
        
        loadMonitoring(url) {
            // Show modal
            this.modal.show();
            
            // Show loader, hide iframe
            this.showLoader();
            
            // Small delay before loading for smooth UX
            setTimeout(() => {
                if (this.iframe) {
                    this.iframe.src = url;
                }
            }, 100);
        }
        
        showLoader() {
            if (this.loader) {
                this.loader.style.display = 'block';
            }
            if (this.iframe) {
                this.iframe.style.display = 'none';
                this.iframe.src = '';
            }
        }
        
        hideLoader() {
            if (this.loader) {
                this.loader.style.display = 'none';
            }
            if (this.iframe) {
                this.iframe.style.display = 'block';
            }
        }
        
        onIframeLoad() {
            this.hideLoader();
            
            // Optional: Communicate with iframe content
            try {
                const iframeWindow = this.iframe.contentWindow;
                // You can add custom communication here if needed
            } catch (e) {
                // Cross-origin restrictions may apply
                console.debug('JDownloads Monitoring: Unable to access iframe content', e);
            }
        }
        
        onIframeError() {
            this.hideLoader();
            
            // Show error message
            if (this.loader) {
                this.loader.innerHTML = `
                    <div class="alert alert-danger m-4">
                        <i class="icon-warning"></i>
                        <strong>Error:</strong> Unable to load monitoring content.
                        Please check your settings and try again.
                    </div>
                `;
                this.loader.style.display = 'block';
            }
        }
        
        resetModal() {
            if (this.iframe) {
                this.iframe.src = '';
            }
            this.showLoader();
            
            // Reset loader content
            if (this.loader) {
                this.loader.innerHTML = `
                    <div class="spinner-border text-primary" role="status" style="width: 3rem; height: 3rem;">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-3">${this.config.loadingText || 'Loading...'}</p>
                `;
            }
        }
        
        escapeHtml(text) {
            const map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, m => map[m]);
        }
    }
    
    // Auto-initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            if (window.jdMonitoringConfig) {
                new JDownloadsMonitoring(window.jdMonitoringConfig);
            }
        });
    } else {
        if (window.jdMonitoringConfig) {
            new JDownloadsMonitoring(window.jdMonitoringConfig);
        }
    }
    
    // Export for manual initialization if needed
    window.JDownloadsMonitoring = JDownloadsMonitoring;
    
})(document, window);
