<?php
/**
 * @package jDownloads
 * @version 4.0  
 * @copyright (C) 2007 - 2021 - Arno Betz - www.jdownloads.com
 * @license http://www.gnu.org/copyleft/gpl.html GNU/GPL, see LICENSE.txt
 * 
 * jDownloads is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

// no direct access
defined('_JEXEC') or die('Restricted Access');
 
use Joomla\CMS\Language\Text;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;

?>

<form action="" method="post" id="adminForm" name="adminForm">
	
    <?php if (!empty($this->sidebar)) : ?>
        <div id="j-sidebar-container" class="col-2">
            <?php echo $this->sidebar; ?>
        </div>
        <div id="j-main-container" class="col-10">
    <?php else : ?>
        <div id="j-main-container">
    <?php endif;?>    
    
    <h1 class="page-title mb-4">
        <i class="fas fa-cloud-upload-alt"></i> <?php echo Text::_('COM_JDOWNLOADS_UPLOADER'); ?>
    </h1>
    
    <div class="alert alert-info">
        <i class="fas fa-info-circle"></i>
        <?php echo Text::_('COM_JDOWNLOADS_UPLOADER_DESC'); ?>
        <?php if ($this->validationMode === 'whitelist') : ?>
            <br><strong><?php echo Text::_('COM_JDOWNLOADS_UPLOADER_ALLOWED_TYPES'); ?>:</strong> <?php echo htmlspecialchars($this->allowedExtensions); ?>
        <?php else : ?>
            <br><strong><?php echo Text::_('COM_JDOWNLOADS_UPLOADER_BLOCKED_TYPES'); ?>:</strong> <?php echo htmlspecialchars($this->blockedExtensions); ?>
        <?php endif; ?>
    </div>
    
    <!-- Upload Manager Container -->
    <div style="background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">
        
        <!-- Upload Area -->
        <div class="upload-area mb-4" style="
            border: 2px dashed #007bff;
            border-radius: 8px;
            padding: 40px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            background-color: #f8f9fa;
        " id="uploadArea">
            <i class="fas fa-cloud-upload-alt" style="font-size: 48px; color: #007bff; margin-bottom: 10px;"></i>
            <h5 class="mt-3"><?php echo Text::_('COM_JDOWNLOADS_UPLOAD_DRAGDROP'); ?></h5>
            <p class="text-muted small"><?php echo Text::_('COM_JDOWNLOADS_UPLOADER_MAX_FILE_SIZE'); ?>:</strong> <?php echo $this->maxFileSize; ?> MB</p>
            <input type="file" id="fileInput" style="display: none;" multiple>
        </div>
        
        <!-- File List -->
        <div id="uploadFiles"></div>
        
        <!-- Summary -->
        <div id="summary" style="display: none; margin-top: 30px; padding: 15px; background: #d4edda; border-radius: 6px; border: 1px solid #c3e6cb; color: #155724;"></div>
        
        <!-- Upload Button -->
        <button class="btn btn-primary btn-lg" id="uploadBtn" style="display: none; width: 100%; margin-top: 20px;">
            <i class="fas fa-upload"></i> <?php echo Text::_('COM_JDOWNLOADS_UPLOAD_ALL'); ?>
        </button>
    </div>
    
    <!-- Form Token -->
    <input type="hidden" name="task" value="" />
    <input type="hidden" name="<?php echo Session::getFormToken(); ?>" value="1" />
    
</div>
</form>
<!-- CSS styles -->
<style>
    .upload-area:hover {
        border-color: #0056b3;
        background-color: #e7f1ff;
    }
    
    .upload-area.dragover {
        border-color: #0056b3;
        background-color: #e7f1ff;
        box-shadow: inset 0 0 10px rgba(0,86,179,0.2);
    }
    
    .file-item {
        background: #f8f9fa;
        border-radius: 6px;
        padding: 15px;
        margin-bottom: 15px;
        display: flex;
        align-items: center;
        gap: 15px;
        border: 1px solid #dee2e6;
    }
    
    .file-item.uploading {
        background: #e7f1ff;
    }
    
    .file-item.success {
        background: #d4edda;
        border-color: #c3e6cb;
    }
    
    .file-item.error {
        background: #f8d7da;
        border-color: #f5c6cb;
    }
    
    .file-icon {
        font-size: 32px;
        color: #007bff;
        width: 50px;
        text-align: center;
        flex-shrink: 0;
    }
    
    .file-info {
        flex: 1;
        min-width: 0;
    }
    
    .file-name {
        font-weight: 600;
        margin-bottom: 5px;
        word-break: break-all;
    }
    
    .file-size {
        font-size: 12px;
        color: #666;
        margin-bottom: 8px;
    }
    
    .progress-container {
        height: 4px;
        margin-top: 8px;
        display: none;
        background: #e9ecef;
        border-radius: 3px;
        overflow: hidden;
    }
    
    .file-item.uploading .progress-container {
        display: block;
    }
    
    .progress-bar {
        height: 100%;
        background: linear-gradient(90deg, #007bff, #0056b3);
        width: 0%;
        transition: width 0.3s ease;
    }
    
    .file-status {
        text-align: right;
        flex-shrink: 1;
        min-width: 60px;
        max-width: 150px;
    }
    
    .file-status-text {
        font-size: 12px;
        color: #666;
        word-break: break-word;
        overflow-wrap: break-word;
        line-height: 1.3;
    }
    
    .btn-remove {
        background: none;
        border: none;
        color: #dc3545;
        cursor: pointer;
        padding: 5px 10px;
        font-size: 18px;
    }
    
    .btn-remove:hover {
        color: #c82333;
    }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const uploadArea = document.getElementById('uploadArea');
    const fileInput = document.getElementById('fileInput');
    const uploadFiles = document.getElementById('uploadFiles');
    const uploadBtn = document.getElementById('uploadBtn');
    const summary = document.getElementById('summary');
    
    let files = [];
    let uploading = false;
    const uploadUrl = 'index.php?option=com_jdownloads&task=uploads.upload&<?php echo Session::getFormToken(); ?>=1&format=json';
    const enableLog = <?php echo $this->enableLog ? 'true' : 'false'; ?>;
    
    // Logging function
    function log(message, data = null) {
        if (!enableLog) return;
        const logDiv = document.getElementById('log');
        if (!logDiv) return;
        
        const timestamp = new Date().toLocaleTimeString();
        const logEntry = document.createElement('div');
        logEntry.style.cssText = 'padding: 5px; margin: 2px 0; background: #f8f9fa; border-left: 3px solid #007bff; font-family: monospace; font-size: 12px;';
        
        let logText = `[${timestamp}] ${message}`;
        if (data) {
            logText += ': ' + JSON.stringify(data, null, 2);
        }
        
        logEntry.textContent = logText;
        logDiv.insertBefore(logEntry, logDiv.firstChild);
    }
    
    // Upload Area Events
    uploadArea.addEventListener('click', () => fileInput.click());
    
    uploadArea.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadArea.classList.add('dragover');
    });
    
    uploadArea.addEventListener('dragleave', () => {
        uploadArea.classList.remove('dragover');
    });
    
    uploadArea.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadArea.classList.remove('dragover');
        addFiles(e.dataTransfer.files);
    });
    
    fileInput.addEventListener('change', (e) => addFiles(e.target.files));
    uploadBtn.addEventListener('click', uploadAllFiles);
    
    function addFiles(fileList) {
        log('Files added', { count: fileList.length });
        for (let file of fileList) {
            const fileId = 'file_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
            const fileObj = { id: fileId, file: file, status: 'pending' };
            files.push(fileObj);
            renderFile(fileObj);
            log('File queued', { name: file.name, size: file.size, id: fileId });
        }
        updateUploadButton();
    }
    
    function renderFile(fileObj) {
        const fileItem = document.createElement('div');
        fileItem.id = fileObj.id;
        fileItem.className = 'file-item';
        fileItem.innerHTML = `
            <div class="file-icon">
                <i class="fas fa-file"></i>
            </div>
            <div class="file-info">
                <div class="file-name">${escapeHtml(fileObj.file.name)}</div>
                <div class="file-size">${formatBytes(fileObj.file.size)}</div>
                <div class="progress-container">
                    <div class="progress-bar"></div>
                </div>
            </div>
            <div class="file-status">
                <button class="btn-remove" title="<?php echo Text::_('COM_JDOWNLOADS_REMOVE'); ?>">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        fileItem.querySelector('.btn-remove').addEventListener('click', (e) => {
            e.preventDefault();
            removeFile(fileObj.id);
        });
        uploadFiles.appendChild(fileItem);
    }
    
    function removeFile(fileId) {
        files = files.filter(f => f.id !== fileId);
        document.getElementById(fileId)?.remove();
        updateUploadButton();
    }
    
    function updateUploadButton() {
        uploadBtn.style.display = (files.length > 0 && !uploading) ? 'block' : 'none';
    }
    
    async function uploadAllFiles() {
        if (uploading || files.length === 0) return;
        
        uploading = true;
        uploadBtn.disabled = true;
        summary.style.display = 'none';
        
        let successCount = 0;
        let errorCount = 0;
        
        for (let fileObj of files) {
            if (fileObj.status === 'pending') {
                const success = await uploadFile(fileObj);
                if (success) successCount++;
                else errorCount++;
            }
        }
        
        showSummary(successCount, errorCount);
        uploading = false;
        uploadBtn.disabled = false;
        updateUploadButton();
    }
    
    async function uploadFile(fileObj) {
        const fileElement = document.getElementById(fileObj.id);
        fileElement.classList.add('uploading');
        const progressBar = fileElement.querySelector('.progress-bar');
        
        log('Upload started', { name: fileObj.file.name, size: fileObj.file.size });
        
        const formData = new FormData();
        formData.append('file', fileObj.file);
        formData.append('<?php echo Session::getFormToken(); ?>', '1');
        
        return new Promise((resolve) => {
            const xhr = new XMLHttpRequest();
            
            // Upload progress
            xhr.upload.addEventListener('progress', (e) => {
                if (e.lengthComputable) {
                    const percentComplete = (e.loaded / e.total) * 100;
                    progressBar.style.width = percentComplete + '%';
                    log('Upload progress', { 
                        name: fileObj.file.name, 
                        loaded: e.loaded, 
                        total: e.total, 
                        percent: Math.round(percentComplete) 
                    });
                }
            });
            
            // Upload complete
            xhr.addEventListener('load', () => {
                log('Response received', { status: xhr.status, statusText: xhr.statusText });
                
                try {
                    const result = JSON.parse(xhr.responseText);
                    log('Response parsed', result);
                    
                    // Check Plupload-format response: error=0 means success, error=1 means failure
                    if (result.error === 0) {
                        fileObj.status = 'success';
                        fileElement.classList.remove('uploading');
                        fileElement.classList.add('success');
                        progressBar.style.width = '100%';
                        fileElement.querySelector('.file-status').innerHTML = '<i class="fas fa-check" style="color: #28a745; font-size: 20px;"></i>';
                        log('Upload SUCCESS', { name: fileObj.file.name });
                        resolve(true);
                    } else {
                        throw new Error(result.msg || 'Upload failed');
                    }
                } catch (error) {
                    log('Upload ERROR', { name: fileObj.file.name, error: error.message });
                    fileObj.status = 'error';
                    fileElement.classList.remove('uploading');
                    fileElement.classList.add('error');
                    fileElement.querySelector('.file-status').innerHTML = '<div class="file-status-text" style="color: #dc3545;">' + escapeHtml(error.message) + '</div>';
                    resolve(false);
                }
            });
            
            // Upload error
            xhr.addEventListener('error', () => {
                log('Upload ERROR', { name: fileObj.file.name, error: 'Network error' });
                fileObj.status = 'error';
                fileElement.classList.remove('uploading');
                fileElement.classList.add('error');
                fileElement.querySelector('.file-status').innerHTML = '<div class="file-status-text" style="color: #dc3545;">Network error</div>';
                resolve(false);
            });
            
            // Upload timeout
            xhr.addEventListener('timeout', () => {
                log('Upload ERROR', { name: fileObj.file.name, error: 'Timeout' });
                fileObj.status = 'error';
                fileElement.classList.remove('uploading');
                fileElement.classList.add('error');
                fileElement.querySelector('.file-status').innerHTML = '<div class="file-status-text" style="color: #dc3545;">Timeout</div>';
                resolve(false);
            });
            
            log('XHR request to', uploadUrl);
            xhr.open('POST', uploadUrl, true);
            xhr.timeout = 300000; // 5 minutes timeout
            xhr.send(formData);
        });
    }
    
    function showSummary(successCount, errorCount) {
        const total = successCount + errorCount;
        let html = `<strong>${total} <?php echo Text::_('COM_JDOWNLOADS_FILES'); ?> <?php echo Text::_('COM_JDOWNLOADS_UPLOAD_PROCESSED'); ?>:</strong><br>`;
        html += `✓ ${successCount} <?php echo Text::_('COM_JDOWNLOADS_SUCCESSFUL'); ?>`;
        
        if (errorCount > 0) {
            html += `<br>✗ ${errorCount} <?php echo Text::_('JERROR_AN_ERROR_HAS_OCCURRED'); ?>`;
            summary.style.background = '#f8d7da';
            summary.style.borderColor = '#f5c6cb';
            summary.style.color = '#721c24';
        } else {
            summary.style.background = '#d4edda';
            summary.style.borderColor = '#c3e6cb';
            summary.style.color = '#155724';
        }
        
        summary.innerHTML = html;
        summary.style.display = 'block';
    }
    
    function formatBytes(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }
    
    function escapeHtml(text) {
        const map = {'&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#039;'};
        return text.replace(/[&<>"']/g, m => map[m]);
    }
});
</script>

<?php if($this->enableLog) : ?>
<div style="margin-top: 30px;">
    <button type="button" class="btn btn-outline-info btn-sm" onclick="document.getElementById('log').style.display = document.getElementById('log').style.display === 'none' ? 'block' : 'none';">
        <i class="fas fa-list"></i> <?php echo Text::_('COM_JDOWNLOADS_UPLOADER_LOG_BTN'); ?>
    </button>
    <button type="button" class="btn btn-outline-secondary btn-sm" onclick="document.getElementById('log').innerHTML = '';">
        <i class="fas fa-eraser"></i> <?php echo Text::_('COM_JDOWNLOADS_UPLOADER_LOG_CLEAR'); ?>
    </button>
    <div id="log" style="display: none; margin-top: 10px; min-height: 150px; height: 200px; overflow-y: auto; border: 1px solid #dee2e6; padding: 10px; background: white; border-radius: 4px; resize: vertical;"></div>
</div>
<?php endif; ?>