<?php
/**
 * @package jDownloads
 * @version 4.0  
 * @copyright (C) 2007 - 2022 - Arno Betz - www.jdownloads.com
 * @license http://www.gnu.org/copyleft/gpl.html GNU/GPL, see LICENSE.txt
 * 
 * jDownloads is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
# Parts from this script are original from the component com_mediamu and are only modified to use it with jDownloads: 
# ------------------------------------------------------------------------
@author Ljubisa - ljufisha.blogspot.com
@copyright Copyright (C) 2012 ljufisha.blogspot.com. All Rights Reserved.
@license - http://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
Technical Support: http://ljufisha.blogspot.com
*/

namespace JDownloads\Component\JDownloads\Administrator\Controller; 

\defined( '_JEXEC' ) or die;

define("COM_MEDIAMU_DEBUG", false);

use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Controller\AdminController;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Session\Session;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\Filesystem\Path;

use JDownloads\Component\JDownloads\Administrator\Helper\JDownloadsHelper;

/**
 * jDownloads Upload Controller
 *
 */
class UploadsController extends AdminController
{
	
    /**
     * Constructor
     *
    */
    function __construct()
    {
        parent::__construct();
    }
    
    public function files() 
    {
         // set redirect
         $this->setRedirect( 'index.php?option=com_jdownloads&view=files');        
    }  
    
    public function downloads() 
    {
         // set redirect
         $this->setRedirect( 'index.php?option=com_jdownloads&view=downloads');        
    }
        
    /**
     * 
     * Set the JSON response and exists script
     * 
     * @param int $code Error Code
     * @param string $msg Error Message
     * @param bool $error
     */
    private function _setResponse($code, $msg = null, $error = true) 
    {
        if($error) 
        {
            $jsonrpc = array (
                "error"     => 1,
                "code"      => $code,
                "msg"       => $msg
            );
        } 
        else 
        {
            $jsonrpc = array (
                "error"     => 0,
                "code"      => $code,
                "msg"       => $msg
            );
        }
        
        die(json_encode($jsonrpc));
        
    }
    
    /**
     * Modern HTML5 file upload handler
     * 
     * @return string JSON response
     */
    public function upload()
    {
        $params = ComponentHelper::getParams('com_jdownloads');
        $app = Factory::getApplication();
        
        // 5 minutes execution time
        @set_time_limit(5 * 60);
        
        // Disable error reporting for clean JSON output
        error_reporting(0);
        
        $session = $app->getSession();
        $user = $app->getIdentity();
        
        // Check token
        if (!$session->checkToken('request')) {
            $this->_setResponse(400, Text::_('JINVALID_TOKEN'));
        }
        
        // Check user permissions
        if (!$user->authorise('core.create', 'com_jdownloads')) {
            $this->_setResponse(400, Text::_('COM_JDOWNLOADS_ERROR_PERM_DENIDED'));
        }
        
        // Check if file was uploaded
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            $this->_setResponse(100, Text::_('COM_JDOWNLOADS_ERROR_UPLOAD_FILE'));
        }
        
        $file = $_FILES['file'];
        
        // Get upload directory from params
        $files_uploaddir = $params->get('files_uploaddir');
        if (empty($files_uploaddir)) {
            $this->_setResponse(100, Text::_('COM_JDOWNLOADS_ERROR_UPLOAD_INVALID_PATH'));
        }
        
        // Check and sanitize target directory
        $targetDir = Path::check($files_uploaddir . '/');
        
        if (!file_exists($targetDir) || !is_dir($targetDir)) {
            $this->_setResponse(100, Text::_('COM_JDOWNLOADS_ERROR_UPLOAD_INVALID_PATH'));
        }
        
        // Get max file size (in MB, convert to bytes)
        $maxFileSizeMB = (int)$params->get('plupload_max_file_size', 512);
        $maxFileSize = $maxFileSizeMB * 1024 * 1024;
        
        // Check file size
        if ($file['size'] > $maxFileSize) {
            $this->_setResponse(100, Text::sprintf('COM_JDOWNLOADS_ERROR_UPLOAD_FILE_TOO_LARGE', $maxFileSizeMB));
        }
        
        // Clean filename using configured folder/file rules (supports UTF-8 when enabled)
        $fileName = JDownloadsHelper::getCleanFolderFileName(basename($file['name']));
        $fileExt = strtolower(File::getExt($fileName));
        
        // Get validation mode (whitelist or blacklist)
        $validationMode = $params->get('upload_validation_mode', 'blacklist');
        
        if ($validationMode === 'whitelist') {
            // Whitelist: Only allow specified extensions
            $allowedExts = $params->get('upload_allowed_extensions', 'zip,rar,pdf,doc,docx,txt,jpg,jpeg,png,gif');
            $allowedExtsArray = array_map('trim', array_map('strtolower', explode(',', $allowedExts)));
            
            if (!in_array($fileExt, $allowedExtsArray)) {
                $this->_setResponse(100, Text::_('COM_JDOWNLOADS_ERROR_UPLOAD_INVALID_FILE_EXTENSION'));
            }
        } else {
            // Blacklist: Block dangerous extensions
            $blockedExts = $params->get('upload_blocked_extensions', 'php,phtml,php3,php4,php5,phps,sh,bat,cmd,com,exe,dll,scr,cpl,vbs,js,jar');
            $blockedExtsArray = array_map('trim', array_map('strtolower', explode(',', $blockedExts)));
            
            if (in_array($fileExt, $blockedExtsArray)) {
                $this->_setResponse(100, Text::_('COM_JDOWNLOADS_ERROR_UPLOAD_BLOCKED_FILE_EXTENSION'));
            }
        }
        
        // Generate unique filename if file exists
        $filePath = $targetDir . $fileName;
        if (file_exists($filePath)) {
            $nameWithoutExt = File::stripExt($fileName);
            $count = 1;
            while (file_exists($targetDir . $nameWithoutExt . '_' . $count . '.' . $fileExt)) {
                $count++;
            }
            $fileName = $nameWithoutExt . '_' . $count . '.' . $fileExt;
            $filePath = $targetDir . $fileName;
        }
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $filePath)) {
            $this->_setResponse(102, Text::_('COM_JDOWNLOADS_ERROR_UPLOAD_MOVE_FAILED'));
        }
        
        // Success
        $this->_setResponse(0, Text::sprintf('COM_JDOWNLOADS_UPLOAD_SUCCESS', $fileName), false);
    }
 
}
?>