<?php
/**
 * jDownloads Upload View Integration
 * 
 * Nutze diese Datei als Ersatz für die aktuelle Uploads-View
 * Platziere sie in: administrator/components/com_jdownloads/tmpl/uploads/
 */

defined('_JEXEC') or die;

use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Factory;

$doc = Factory::getDocument();
$doc->getWebAssetManager()
    ->useStyle('bootstrap.css')
    ->useScript('bootstrap.bundle.min.js');
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <h1 class="page-title mb-4">
                <i class="fas fa-upload"></i> File Upload
            </h1>
            
            <!-- Upload Manager -->
            <div id="uploadManager" style="background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">
                
                <!-- Upload Area -->
                <div class="upload-area mb-4" style="
                    border: 2px dashed #007bff;
                    border-radius: 8px;
                    padding: 40px;
                    text-align: center;
                    cursor: pointer;
                    transition: all 0.3s ease;
                    background-color: #f8f9fa;
                " id="uploadArea">
                    <i class="fas fa-cloud-upload-alt" style="font-size: 48px; color: #007bff; margin-bottom: 10px;"></i>
                    <h5 class="mt-3">Dateien hier ziehen oder klicken zum Auswählen</h5>
                    <p class="text-muted small">Unterstützte Formate: ZIP, RAR, 7Z, TAR, GZ, PDF, TXT, DOC, XLS, etc.</p>
                    <input type="file" id="fileInput" style="display: none;" multiple>
                </div>
                
                <!-- File List -->
                <div id="uploadFiles"></div>
                
                <!-- Summary -->
                <div id="summary" style="display: none; margin-top: 30px; padding: 15px; background: #d4edda; border-radius: 6px; border: 1px solid #c3e6cb; color: #155724;"></div>
                
                <!-- Upload Button -->
                <button class="btn btn-primary btn-lg" id="uploadBtn" style="display: none; width: 100%; margin-top: 20px;">
                    <i class="fas fa-upload"></i> Alle Dateien hochladen
                </button>
            </div>
        </div>
    </div>
</div>

<style>
    .upload-area:hover {
        border-color: #0056b3;
        background-color: #e7f1ff;
    }
    
    .upload-area.dragover {
        border-color: #0056b3;
        background-color: #e7f1ff;
        box-shadow: inset 0 0 10px rgba(0,86,179,0.2);
    }
    
    .file-item {
        background: #f8f9fa;
        border-radius: 6px;
        padding: 15px;
        margin-bottom: 15px;
        display: flex;
        align-items: center;
        gap: 15px;
        border: 1px solid #dee2e6;
    }
    
    .file-item.uploading {
        background: #e7f1ff;
    }
    
    .file-item.success {
        background: #d4edda;
        border-color: #c3e6cb;
    }
    
    .file-item.error {
        background: #f8d7da;
        border-color: #f5c6cb;
    }
    
    .file-icon {
        font-size: 32px;
        color: #007bff;
        width: 50px;
        text-align: center;
        flex-shrink: 0;
    }
    
    .file-info {
        flex: 1;
        min-width: 0;
    }
    
    .file-name {
        font-weight: 600;
        margin-bottom: 5px;
        word-break: break-all;
    }
    
    .file-size {
        font-size: 12px;
        color: #666;
        margin-bottom: 8px;
    }
    
    .progress-container {
        height: 4px;
        margin-top: 8px;
        display: none;
        background: #e9ecef;
        border-radius: 3px;
        overflow: hidden;
    }
    
    .file-item.uploading .progress-container {
        display: block;
    }
    
    .progress-bar {
        height: 100%;
        background: linear-gradient(90deg, #007bff, #0056b3);
        width: 0%;
        transition: width 0.3s ease;
    }
    
    .file-status {
        text-align: right;
        white-space: nowrap;
        flex-shrink: 0;
        min-width: 100px;
    }
    
    .file-status-text {
        font-size: 12px;
        color: #666;
    }
    
    .btn-remove {
        background: none;
        border: none;
        color: #dc3545;
        cursor: pointer;
        padding: 5px 10px;
        font-size: 18px;
    }
    
    .btn-remove:hover {
        color: #c82333;
    }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const uploadArea = document.getElementById('uploadArea');
    const fileInput = document.getElementById('fileInput');
    const uploadFiles = document.getElementById('uploadFiles');
    const uploadBtn = document.getElementById('uploadBtn');
    const summary = document.getElementById('summary');
    
    let files = [];
    let uploading = false;
    
    // Upload Area Events
    uploadArea.addEventListener('click', () => fileInput.click());
    
    uploadArea.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadArea.classList.add('dragover');
    });
    
    uploadArea.addEventListener('dragleave', () => {
        uploadArea.classList.remove('dragover');
    });
    
    uploadArea.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadArea.classList.remove('dragover');
        addFiles(e.dataTransfer.files);
    });
    
    fileInput.addEventListener('change', (e) => addFiles(e.target.files));
    uploadBtn.addEventListener('click', uploadAllFiles);
    
    function addFiles(fileList) {
        for (let file of fileList) {
            const fileId = 'file_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
            const fileObj = { id: fileId, file: file, status: 'pending' };
            files.push(fileObj);
            renderFile(fileObj);
        }
        updateUploadButton();
    }
    
    function renderFile(fileObj) {
        const fileItem = document.createElement('div');
        fileItem.id = fileObj.id;
        fileItem.className = 'file-item';
        fileItem.innerHTML = `
            <div class="file-icon">
                <i class="fas fa-file"></i>
            </div>
            <div class="file-info">
                <div class="file-name">${escapeHtml(fileObj.file.name)}</div>
                <div class="file-size">${formatBytes(fileObj.file.size)}</div>
                <div class="progress-container">
                    <div class="progress-bar"></div>
                </div>
            </div>
            <div class="file-status">
                <button class="btn-remove" title="Entfernen">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        fileItem.querySelector('.btn-remove').addEventListener('click', () => removeFile(fileObj.id));
        uploadFiles.appendChild(fileItem);
    }
    
    function removeFile(fileId) {
        files = files.filter(f => f.id !== fileId);
        document.getElementById(fileId)?.remove();
        updateUploadButton();
    }
    
    function updateUploadButton() {
        uploadBtn.style.display = (files.length > 0 && !uploading) ? 'block' : 'none';
    }
    
    async function uploadAllFiles() {
        if (uploading || files.length === 0) return;
        
        uploading = true;
        uploadBtn.disabled = true;
        summary.style.display = 'none';
        
        let successCount = 0;
        let errorCount = 0;
        
        for (let fileObj of files) {
            if (fileObj.status === 'pending') {
                const success = await uploadFile(fileObj);
                if (success) successCount++;
                else errorCount++;
            }
        }
        
        showSummary(successCount, errorCount);
        uploading = false;
        uploadBtn.disabled = false;
        updateUploadButton();
    }
    
    async function uploadFile(fileObj) {
        const fileElement = document.getElementById(fileObj.id);
        fileElement.classList.add('uploading');
        
        const formData = new FormData();
        formData.append('file', fileObj.file);
        
        try {
            const response = await fetch('upload-handler.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                fileObj.status = 'success';
                fileElement.classList.remove('uploading');
                fileElement.classList.add('success');
                fileElement.querySelector('.progress-bar').style.width = '100%';
                fileElement.querySelector('.file-status').innerHTML = '<i class="fas fa-check" style="color: #28a745;"></i>';
                return true;
            } else {
                throw new Error(result.error || 'Upload failed');
            }
        } catch (error) {
            fileObj.status = 'error';
            fileElement.classList.remove('uploading');
            fileElement.classList.add('error');
            fileElement.querySelector('.file-status').innerHTML = '<div class="file-status-text" style="color: #dc3545;">' + escapeHtml(error.message) + '</div>';
            return false;
        }
    }
    
    function showSummary(successCount, errorCount) {
        const total = successCount + errorCount;
        let html = `<strong>${total} Datei(en) verarbeitet:</strong><br>`;
        html += `✓ ${successCount} erfolgreich`;
        
        if (errorCount > 0) {
            html += `<br>✗ ${errorCount} Fehler`;
            summary.style.background = '#f8d7da';
            summary.style.borderColor = '#f5c6cb';
            summary.style.color = '#721c24';
        }
        
        summary.innerHTML = html;
        summary.style.display = 'block';
    }
    
    function formatBytes(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }
    
    function escapeHtml(text) {
        const map = {'&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#039;'};
        return text.replace(/[&<>"']/g, m => map[m]);
    }
});
</script>
